import httpx, asyncio, os, logging, re, mimetypes
from bs4 import BeautifulSoup
from telegram import Update, BotCommand
from telegram.ext import Application, CommandHandler, MessageHandler, filters, CallbackContext
from telegram.constants import ChatAction
from urllib.parse import urlparse

# Configuración
logging.basicConfig(format='%(asctime)s - %(name)s - %(levelname)s - %(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)

TELEGRAM_BOT_TOKEN = "7971082679:AAFNvmUMFjQ2utlessC2-b-yjiFBN-c16bA"
UPLOAD_FOLDER = "downloads"
os.makedirs(UPLOAD_FOLDER, exist_ok=True)

class File:
    def __init__(self, name=None, fileID=None, download_url=None):
        self.name = name
        self.id = fileID
        self.download_url = download_url

class Response:
    SuccessLogin = "SuccessLogin"
    FailedLogin = "FailedLogin"
    SuccessSendFile = "SuccessSendFile"
    FailedSendFile = "FailedSendFile"
    SuccessDeleteFile = "SuccessDeleteFile"
    FailedDeleteFile = "FailedDeleteFile"
    NoExistFile = "NoExistFile"
    SuccessDownloadFile = "SuccessDownloadFile"
    FailedDownloadFile = "FailedDownloadFile"
    SuccessDownloadLink = "SuccessDownloadLink"

class WebClient:
    def __init__(self):
        self.url = "https://talleres.ucf.edu.cu"
        self.node = 49
        self.username = "luisernestorb95"
        self.password = "Luisito1995*"
        self.connected = False
        self.sess = httpx.AsyncClient(timeout=None)
        self.csrfToken = None

    async def connect(self):
        try:
            form = await self.sess.get(f"{self.url}/login/indico/form")
            soup = BeautifulSoup(form.json()["html"], "html.parser")
            self.csrfToken = soup.find('input', {'name':'csrf_token'})['value']
            
            login = await self.sess.post(
                f"{self.url}/login", 
                data={
                    "csrf_token": self.csrfToken,
                    "_provider": "indico",
                    "identifier": self.username,
                    "password": self.password
                },
                follow_redirects=True
            )
            
            if "Cerrar sesión" in login.text:
                self.connected = True
                return Response.SuccessLogin
            return Response.FailedLogin
        except Exception as e:
            logger.error(f"Error en conexión: {e}")
            return Response.FailedLogin

    async def sendFile(self, path):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedSendFile}
        
        try:
            with open(path, 'rb') as file:
                up = await self.sess.post(
                    f"{self.url}/event/{self.node}/manage/attachments/add/files",
                    data={
                        "csrf_token": self.csrfToken,
                        "__file_change_trigger": "added-file",
                        "folder": "__None",
                        "acl": []
                    },
                    files={"files": file},
                    follow_redirects=True
                )
            
            if up.json()['success']:
                soup = BeautifulSoup(up.json()["attachment_list"], 'html.parser')
                attachments = soup.select('a.attachment[data-attachment-id]')
                listFiles = []
                
                for a in attachments:
                    name = a.get_text(strip=True)
                    fileID = a.get('data-attachment-id')
                    if name and fileID:
                        download_url = f"{self.url}/event/{self.node}/attachments/202/{fileID}/file"
                        listFiles.append(File(name=name, fileID=fileID, download_url=download_url))
                
                if listFiles:
                    return {"file": listFiles[-1], "listFiles": listFiles, "response": Response.SuccessSendFile}
            
            return {"response": Response.FailedSendFile}
        except Exception as e:
            logger.error(f"Error subiendo archivo: {e}")
            return {"response": Response.FailedSendFile}

    async def deleteFile(self, identifier):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return Response.FailedDeleteFile
        
        try:
            # Extraer ID
            fileID = None
            if identifier.isdigit():
                fileID = identifier
            elif '/attachments/' in identifier:
                parts = identifier.split('/')
                for i, part in enumerate(parts):
                    if part == 'attachments' and i + 2 < len(parts):
                        fileID = parts[i + 2]
                        break
            
            if not fileID:
                return Response.NoExistFile
            
            self.sess.headers.update({"X-Csrf-Token": self.csrfToken})
            dele = await self.sess.delete(
                f"{self.url}/event/{self.node}/manage/attachments/202/{fileID}/",
                follow_redirects=True
            )
            
            if "No se ha podido encontrar el ítem" in dele.text:
                return Response.NoExistFile
            
            if dele.json()['success']:
                return Response.SuccessDeleteFile
            
            return Response.FailedDeleteFile
        except Exception as e:
            logger.error(f"Error eliminando archivo: {e}")
            return Response.FailedDeleteFile

    async def downloadFile(self, identifier):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedDownloadFile}
        
        try:
            # Extraer ID
            fileID = None
            if identifier.isdigit():
                fileID = identifier
            elif '/attachments/' in identifier:
                parts = identifier.split('/')
                for i, part in enumerate(parts):
                    if part == 'attachments' and i + 2 < len(parts):
                        fileID = parts[i + 2]
                        break
            
            if not fileID:
                return {"response": Response.FailedDownloadFile}
            
            download_url = f"{self.url}/event/{self.node}/attachments/202/{fileID}/file"
            
            async with self.sess.stream("GET", download_url, follow_redirects=True) as streamo:
                streamo.raise_for_status()
                
                disposition = streamo.headers.get("Content-Disposition", "")
                if 'attachment' in disposition and 'filename="' in disposition:
                    name = disposition.split('attachment; filename="')[1].split('"')[0]
                else:
                    name = f"archivo_{fileID}"
                
                name = os.path.join(UPLOAD_FOLDER, name)
                
                with open(name, "wb") as wfile:
                    async for chunk in streamo.aiter_bytes(chunk_size=8192):
                        wfile.write(chunk)
                
                return {"file": File(name=name, fileID=fileID, download_url=download_url), 
                        "response": Response.SuccessDownloadFile}
        except Exception as e:
            logger.error(f"Error descargando archivo: {e}")
            return {"response": Response.FailedDownloadFile}

    async def listFiles(self):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedSendFile}
        
        try:
            response = await self.sess.get(
                f"{self.url}/event/{self.node}/manage/attachments",
                follow_redirects=True
            )
            
            soup = BeautifulSoup(response.text, 'html.parser')
            attachments = soup.select('a.attachment[data-attachment-id]')
            listFiles = []
            
            for a in attachments:
                name = a.get_text(strip=True)
                fileID = a.get('data-attachment-id')
                if name and fileID and not name.startswith('data-'):
                    download_url = f"{self.url}/event/{self.node}/attachments/202/{fileID}/file"
                    listFiles.append(File(name=name, fileID=fileID, download_url=download_url))
            
            return {"listFiles": listFiles, "response": Response.SuccessSendFile}
        except Exception as e:
            logger.error(f"Error listando archivos: {e}")
            return {"response": Response.FailedSendFile}

# Descargador de enlaces externos
class LinkDownloader:
    @staticmethod
    def get_filename_from_url(url, content_type=None):
        """Obtener nombre de archivo desde URL"""
        parsed = urlparse(url)
        filename = os.path.basename(parsed.path)
        
        if not filename or filename == '/':
            filename = f"descarga_{hash(url) % 10000}"
        
        # Asegurar extensión
        if '.' not in filename:
            if content_type:
                ext = mimetypes.guess_extension(content_type)
                if ext:
                    filename += ext
        
        return filename
    
    @staticmethod
    async def download_from_link(url):
        """Descargar archivo desde cualquier enlace"""
        try:
            async with httpx.AsyncClient(timeout=30.0) as client:
                # Hacer HEAD request primero para obtener info
                head_response = await client.head(url, follow_redirects=True)
                content_type = head_response.headers.get('Content-Type', '')
                content_length = head_response.headers.get('Content-Length', 0)
                
                filename = LinkDownloader.get_filename_from_url(url, content_type)
                local_path = os.path.join(UPLOAD_FOLDER, filename)
                
                # Descargar archivo
                async with client.stream('GET', url, follow_redirects=True) as response:
                    response.raise_for_status()
                    
                    with open(local_path, 'wb') as f:
                        async for chunk in response.aiter_bytes(chunk_size=8192):
                            f.write(chunk)
                
                return {"file": File(name=local_path, fileID=None, download_url=url), 
                        "response": Response.SuccessDownloadLink}
                
        except Exception as e:
            logger.error(f"Error descargando enlace: {e}")
            return {"response": Response.FailedDownloadFile}

web_client = WebClient()
link_downloader = LinkDownloader()

# Handlers
async def start(update: Update, context: CallbackContext):
    await update.message.reply_text(
        "Bot de Archivos - Versión Completa\n\n"
        "📤 SUBIR ARCHIVOS:\n"
        "- Envía CUALQUIER archivo (APK, ZIP, RAR, MP3, MP4, etc.)\n"
        "- O envía un enlace de descarga directa\n\n"
        "📋 COMANDOS:\n"
        "/list - Ver archivos en la web\n"
        "/delete <id> - Eliminar archivo\n"
        "/download <id> - Descargar archivo\n"
        "/help - Ver ayuda completa\n\n"
        "✅ SOPORTA: APK, APKS, ZIP, RAR, MP3, MP4, AVI, MKV, TS, DOC, PDF, EXE, etc."
    )

async def help_command(update: Update, context: CallbackContext):
    help_text = """📚 AYUDA COMPLETA

📤 SUBIR ARCHIVOS:
1. Envía CUALQUIER archivo al bot:
   • APK, APKS (Aplicaciones Android)
   • ZIP, RAR, 7Z (Archivos comprimidos)
   • MP3, WAV, FLAC (Audio)
   • MP4, AVI, MKV, TS, MOV (Video)
   • PDF, DOC, DOCX, XLS, PPT (Documentos)
   • EXE, MSI (Ejecutables Windows)
   • Y cualquier otro tipo de archivo

2. O envía un enlace de descarga directa:
   • El bot descargará el archivo
   • Lo subirá a la web
   • Te dará el enlace permanente

🔗 ENLACE DE DESCARGA:
• Copia el enlace y ábrelo en tu navegador
• Descarga directa sin login
• Almacenamiento permanente

🗑 ELIMINAR ARCHIVOS:
/delete 12345
/delete https://talleres.ucf.edu.cu/.../12345/file

📥 DESCARGAR ARCHIVOS:
/download 12345
/download https://talleres.ucf.edu.cu/.../12345/file

📁 LISTAR ARCHIVOS:
/list - Ver todos con enlaces

💡 EJEMPLOS:
• Envía un archivo APK → Se sube automáticamente
• Envía enlace a MP4 → Se descarga y sube
• /delete 123 → Elimina archivo ID 123"""
    
    await update.message.reply_text(help_text)

async def list_command(update: Update, context: CallbackContext):
    await update.message.reply_text("Obteniendo lista de archivos...")
    
    result = await web_client.listFiles()
    
    if result["response"] == Response.FailedSendFile:
        await update.message.reply_text("Error de login")
        return
    
    if not result.get("listFiles"):
        await update.message.reply_text("No hay archivos almacenados")
        return
    
    message = "📁 ARCHIVOS ALMACENADOS:\n\n"
    for file in result["listFiles"]:
        message += f"📄 {file.name}\n"
        message += f"🆔 ID: {file.id}\n"
        message += f"🔗 {file.download_url}\n"
        message += f"🗑 /delete {file.id}\n\n"
    
    await update.message.reply_text(message)

async def delete_command(update: Update, context: CallbackContext):
    if not context.args:
        await update.message.reply_text("Uso: /delete <id_o_url>\nEjemplo: /delete 12345")
        return
    
    identifier = context.args[0]
    await update.message.reply_text(f"Eliminando {identifier}...")
    
    result = await web_client.deleteFile(identifier)
    
    if result == Response.SuccessDeleteFile:
        await update.message.reply_text("✅ Archivo eliminado")
    elif result == Response.NoExistFile:
        await update.message.reply_text("❌ Archivo no encontrado")
    else:
        await update.message.reply_text("❌ Error al eliminar")

async def download_command(update: Update, context: CallbackContext):
    if not context.args:
        await update.message.reply_text("Uso: /download <id_o_url>\nEjemplo: /download 12345")
        return
    
    identifier = context.args[0]
    await update.message.reply_text(f"Descargando {identifier}...")
    
    await update.message.chat.send_action(ChatAction.UPLOAD_DOCUMENT)
    
    result = await web_client.downloadFile(identifier)
    
    if result["response"] == Response.SuccessDownloadFile:
        with open(result["file"].name, 'rb') as file:
            await update.message.reply_document(
                document=file,
                caption=f"✅ Descargado\n📝 {result['file'].name}\n🆔 {result['file'].id}\n🔗 {result['file'].download_url}"
            )
        os.remove(result["file"].name)
    else:
        await update.message.reply_text("❌ Error al descargar")

async def handle_message(update: Update, context: CallbackContext):
    """Maneja cualquier mensaje: archivos o enlaces"""
    try:
        # Verificar si es un enlace
        text = update.message.text or update.message.caption or ""
        
        # Detectar si es una URL
        url_patterns = [
            r'https?://[^\s]+',
            r'www\.[^\s]+\.[^\s]+',
            r'ftp://[^\s]+'
        ]
        
        is_url = False
        for pattern in url_patterns:
            if re.search(pattern, text):
                is_url = True
                break
        
        if is_url:
            # Es un enlace de descarga
            url = text.strip()
            await update.message.reply_text(f"🔗 Detectado enlace: {url[:50]}...")
            await update.message.reply_text("Descargando desde enlace externo...")
            
            await update.message.chat.send_action(ChatAction.TYPING)
            
            # Descargar desde enlace
            result = await link_downloader.download_from_link(url)
            
            if result["response"] == Response.SuccessDownloadLink:
                await update.message.reply_text("✅ Descarga completada. Subiendo a la web...")
                
                # Subir a la web
                web_result = await web_client.sendFile(result["file"].name)
                
                if web_result["response"] == Response.SuccessSendFile and web_result.get("file"):
                    message = f"✅ ARCHIVO SUBIDO DESDE ENLACE\n\n"
                    message += f"📝 Nombre: {web_result['file'].name}\n"
                    message += f"🆔 ID: {web_result['file'].id}\n"
                    message += f"🔗 Enlace permanente: {web_result['file'].download_url}\n\n"
                    message += f"🗑 Eliminar: /delete {web_result['file'].id}\n"
                    message += f"📥 Descargar: /download {web_result['file'].id}"
                    
                    await update.message.reply_text(message)
                else:
                    await update.message.reply_text("❌ Error al subir a la web")
                
                # Limpiar archivo temporal
                if os.path.exists(result["file"].name):
                    os.remove(result["file"].name)
            else:
                await update.message.reply_text("❌ Error al descargar desde el enlace")
            
            return
        
        # Si no es URL, verificar si es archivo
        file_obj = None
        file_name = None
        
        # Detectar tipo de archivo
        if update.message.document:
            file_obj = update.message.document
            file_name = file_obj.file_name or f"document_{file_obj.file_id[:8]}"
            
        elif update.message.photo:
            file_obj = update.message.photo[-1]
            file_name = f"photo_{file_obj.file_id[:8]}.jpg"
            
        elif update.message.video:
            file_obj = update.message.video
            file_name = file_obj.file_name or f"video_{file_obj.file_id[:8]}.mp4"
            
        elif update.message.audio:
            file_obj = update.message.audio
            file_name = file_obj.file_name or f"audio_{file_obj.file_id[:8]}.mp3"
            
        elif update.message.voice:
            file_obj = update.message.voice
            file_name = f"voice_{file_obj.file_id[:8]}.ogg"
            
        elif update.message.video_note:
            file_obj = update.message.video_note
            file_name = f"video_note_{file_obj.file_id[:8]}.mp4"
            
        elif update.message.sticker:
            file_obj = update.message.sticker
            file_name = f"sticker_{file_obj.file_id[:8]}.webp"
        
        elif update.message.animation:  # GIFs
            file_obj = update.message.animation
            file_name = file_obj.file_name or f"animation_{file_obj.file_id[:8]}.mp4"
            
        else:
            await update.message.reply_text("Envía un archivo o enlace de descarga. Usa /help para ayuda.")
            return
        
        # Procesar archivo
        await update.message.chat.send_action(ChatAction.TYPING)
        await update.message.reply_text(f"📥 Descargando {file_name}...")
        
        # Descargar de Telegram
        bot = context.bot
        tg_file = await bot.get_file(file_obj.file_id)
        local_path = os.path.join(UPLOAD_FOLDER, file_name)
        
        await tg_file.download_to_drive(local_path)
        
        await update.message.reply_text("☁️ Subiendo a la web...")
        
        # Subir a la web
        result = await web_client.sendFile(local_path)
        
        if result["response"] == Response.SuccessSendFile and result.get("file"):
            # Obtener tamaño del archivo
            file_size = os.path.getsize(local_path)
            size_str = f"{file_size:,} bytes"
            if file_size > 1024*1024:
                size_str = f"{file_size/(1024*1024):.2f} MB"
            elif file_size > 1024:
                size_str = f"{file_size/1024:.2f} KB"
            
            message = f"✅ ARCHIVO SUBIDO EXITOSAMENTE\n\n"
            message += f"📝 Nombre: {result['file'].name}\n"
            message += f"📦 Tamaño: {size_str}\n"
            message += f"🆔 ID: {result['file'].id}\n"
            message += f"🔗 Enlace permanente: {result['file'].download_url}\n\n"
            message += f"🗑 Eliminar: /delete {result['file'].id}\n"
            message += f"📥 Descargar: /download {result['file'].id}"
            
            await update.message.reply_text(message)
        else:
            await update.message.reply_text("❌ Error al subir archivo")
        
        # Limpiar archivo local
        if os.path.exists(local_path):
            os.remove(local_path)
        
    except Exception as e:
        logger.error(f"Error procesando mensaje: {e}")
        await update.message.reply_text("❌ Error procesando. Intenta de nuevo.")

async def set_commands(application: Application):
    commands = [
        BotCommand("start", "Iniciar bot"),
        BotCommand("list", "Ver archivos almacenados"),
        BotCommand("delete", "Eliminar archivo por ID"),
        BotCommand("download", "Descargar archivo por ID"),
        BotCommand("help", "Ayuda completa"),
    ]
    await application.bot.set_my_commands(commands)

def main():
    application = Application.builder().token(TELEGRAM_BOT_TOKEN).build()
    
    # Comandos
    application.add_handler(CommandHandler("start", start))
    application.add_handler(CommandHandler("help", help_command))
    application.add_handler(CommandHandler("list", list_command))
    application.add_handler(CommandHandler("delete", delete_command))
    application.add_handler(CommandHandler("download", download_command))
    
    # Manejador universal para TODO
    application.add_handler(MessageHandler(
        filters.TEXT | filters.Document.ALL | filters.PHOTO | 
        filters.VIDEO | filters.AUDIO | filters.VOICE | 
        filters.VIDEO_NOTE | filters.ANIMATION, 
        handle_message
    ))
    
    application.post_init = set_commands
    application.add_error_handler(lambda u, c: logger.error(f"Error: {c.error}", exc_info=True))
    
    print("="*60)
    print("🤖 BOT DE ARCHIVOS - VERSIÓN COMPLETA")
    print("="*60)
    print("✅ Acepta TODOS los archivos y enlaces")
    print("✅ Extensiones soportadas: APK, ZIP, RAR, MP3, MP4, PDF, DOC, etc.")
    print("✅ Descarga desde cualquier enlace directo")
    print("✅ Listo para recibir archivos...")
    print("="*60)
    
    application.run_polling()

if __name__ == "__main__":
    main()