import httpx, asyncio, os, logging, re, mimetypes
from bs4 import BeautifulSoup
from telegram import Update, BotCommand, InputMediaDocument
from telegram.ext import Application, CommandHandler, MessageHandler, filters, CallbackContext
from telegram.constants import ChatAction
from urllib.parse import urlparse

# --- CONFIGURACIÓN ---
logging.basicConfig(format='%(asctime)s - %(name)s - %(levelname)s - %(message)s', level=logging.INFO)
logger = logging.getLogger(__name__) # Corrected logger name to __name__

TELEGRAM_BOT_TOKEN = "7971082679:AAFNvmUMFjQ2utlessC2-b-yjiFBN-c16bA"
UPLOAD_FOLDER = "downloads"
os.makedirs(UPLOAD_FOLDER, exist_ok=True)

# Tamaño máximo de cada parte en bytes (10 MB)
MAX_PART_SIZE = 10 * 1024 * 1024 

class File:
    def __init__(self, name=None, fileID=None, download_url=None): # Corrected init to __init__
        self.name = name
        self.id = fileID
        self.download_url = download_url

class Response:
    SuccessLogin = "SuccessLogin"
    FailedLogin = "FailedLogin"
    SuccessSendFile = "SuccessSendFile"
    FailedSendFile = "FailedSendFile"
    SuccessDeleteFile = "SuccessDeleteFile"
    FailedDeleteFile = "FailedDeleteFile"
    NoExistFile = "NoExistFile"
    SuccessDownloadFile = "SuccessDownloadFile"
    FailedDownloadFile = "FailedDownloadFile"
    SuccessDownloadLink = "SuccessDownloadLink"

class WebClient:
    def __init__(self): # Corrected init to __init__
        self.url = "https://eventos.uh.cu/"
        self.node = 150
        self.username = "luisernestorb95"
        self.password = "Luisito1995*"
        self.connected = False
        self.sess = httpx.AsyncClient(timeout=None)
        self.csrfToken = None

    async def connect(self):
        try:
            form = await self.sess.get(f"{self.url}/login/indico/form")
            form.raise_for_status() # Raise HTTPStatusError for bad responses (4xx or 5xx)
            
            # Indico devuelve JSON con HTML, no HTML directo
            form_json = form.json()
            if not isinstance(form_json, dict) or "html" not in form_json:
                logger.error("Error: Respuesta de login/indico/form no contiene la clave 'html'.")
                return Response.FailedLogin

            soup = BeautifulSoup(form_json["html"], "html.parser")
            csrf_token_element = soup.find('input', {'name':'csrf_token'})
            if csrf_token_element is None or 'value' not in csrf_token_element.attrs:
                logger.error("Error: No se encontró el token CSRF en la página de login.")
                return Response.FailedLogin
            self.csrfToken = csrf_token_element['value']
            
            login_response = await self.sess.post(
                f"{self.url}/login", 
                data={
                    "csrf_token": self.csrfToken,
                    "_provider": "indico",
                    "identifier": self.username,
                    "password": self.password
                },
                follow_redirects=True
            )
            login_response.raise_for_status() # Raise HTTPStatusError for bad responses (4xx or 5xx)
            
            if "Cerrar sesión" in login_response.text:
                self.connected = True
                logger.info("Conexión exitosa a talleres.ucf.edu.cu")
                return Response.SuccessLogin
            
            logger.warning(f"Login fallido: {login_response.text}")
            return Response.FailedLogin
        except httpx.HTTPStatusError as e:
            logger.error(f"Error HTTP en conexión: {e.response.status_code} - {e}")
            return Response.FailedLogin
        except (httpx.RequestError, ValueError, KeyError) as e:
            logger.error(f"Error en conexión: {e}")
            return Response.FailedLogin

    async def _upload_single_file(self, file_path, filename_on_server):
        """Método interno para subir una única parte/archivo."""
        if not self.connected:
            await self.connect() # Intentar conectar si no lo estamos
            if not self.connected:
                return {"response": Response.FailedSendFile}

        try:
            with open(file_path, 'rb') as file_content:
                # Corregido el formato files para httpx: ("filename", file_object)
                up = await self.sess.post(
                    f"{self.url}/event/{self.node}/manage/attachments/add/files",
                    data={
                        "csrf_token": self.csrfToken,
                        "__file_change_trigger": "added-file",
                        "folder": "__None",
                        "acl": []
                    },
                    files={"files": (filename_on_server, file_content)}, # Usamos el nombre del archivo para el servidor
                    follow_redirects=True
                )
            up.raise_for_status() # Raise HTTPStatusError for bad responses (4xx or 5xx)
            up_json = up.json()

            if up_json.get('success'):
                # Indico devuelve una lista de attachments actualizados,
                # Buscamos el que acabamos de subir por su nombre o id
                soup = BeautifulSoup(up_json["attachment_list"], 'html.parser')
                attachments = soup.select('a.attachment[data-attachment-id]')
                
                # Intentamos encontrar el archivo subido por su nombre
                for a in attachments:
                    name_found = a.get_text(strip=True)
                    if name_found == filename_on_server:
                        fileID = a.get('data-attachment-id')
                        download_url = f"{self.url}/event/{self.node}/attachments/202/{fileID}/file"
                        return {"response": Response.SuccessSendFile, "file": File(name=name_found, fileID=fileID, download_url=download_url)}
                
                logger.warning(f"Archivo subido '{filename_on_server}' pero no encontrado en la lista de attachments devuelta.")
                return {"response": Response.FailedSendFile}
            else:
                logger.warning(f"API respondió con éxito=false al subir {filename_on_server}: {up_json}")
                return {"response": Response.FailedSendFile}
        except httpx.HTTPStatusError as e:
            logger.error(f"Error HTTP al subir parte {filename_on_server}: {e.response.status_code} - {e}")
            return {"response": Response.FailedSendFile}
        except (httpx.RequestError, FileNotFoundError, KeyError, ValueError) as e:
            logger.error(f"Error al enviar la parte {filename_on_server}: {e}")
            return {"response": Response.FailedSendFile}
        except Exception as e:
            logger.error(f"Error inesperado al enviar la parte {filename_on_server}: {e}")
            return {"response": Response.FailedSendFile}

    async def sendFile(self, original_file_path):
        """
        Sube un archivo, cortándolo en partes si es más grande que MAX_PART_SIZE.
        Devuelve una lista de objetos File si se subieron varias partes.
        """
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedSendFile}
        
        file_size = os.path.getsize(original_file_path)
        original_filename = os.path.basename(original_file_path)
        
        uploaded_files_info = []
        
        if file_size <= MAX_PART_SIZE:
            # Si el archivo es pequeño, subirlo completo
            logger.info(f"Subiendo archivo completo: {original_filename}")
            result = await self._upload_single_file(original_file_path, original_filename)
            if result["response"] == Response.SuccessSendFile:
                uploaded_files_info.append(result["file"])
        else:
            # Cortar y subir en partes
            logger.info(f"Archivo {original_filename} es grande ({file_size / (1024*1024):.2f} MB), cortando en partes de {MAX_PART_SIZE / (1024*1024):.0f} MB.")
            part_num = 0
            with open(original_file_path, 'rb') as f:
                while True:
                    data = f.read(MAX_PART_SIZE)
                    if not data:
                        break # No más datos
                    
                    part_num += 1
                    part_filename = f"{os.path.splitext(original_filename)[0]}.part{part_num}{os.path.splitext(original_filename)[1] or ''}"
                    part_path = os.path.join(UPLOAD_FOLDER, part_filename)
                    
                    with open(part_path, 'wb') as part_file:
                        part_file.write(data)
                    
                    logger.info(f"Subiendo parte {part_num}: {part_filename}")
                    part_upload_result = await self._upload_single_file(part_path, part_filename)
                    
                    if part_upload_result["response"] == Response.SuccessSendFile:
                        uploaded_files_info.append(part_upload_result["file"])
                    else:
                        logger.error(f"Fallo al subir la parte {part_num} de {original_filename}")
                        # Podemos decidir si continuar o abortar aquí
                        # Por ahora, continuamos para subir las otras partes si es posible
                        pass 
                    
                    os.remove(part_path) # Limpiar la parte temporal
            
        if uploaded_files_info:
            return {"response": Response.SuccessSendFile, "files": uploaded_files_info}
        else:
            return {"response": Response.FailedSendFile}

    # Resto de los métodos de WebClient (listFiles, deleteFile, downloadFile) sin cambios por ahora
    # ... (El código de listFiles, deleteFile, downloadFile iría aquí, asegurándose de que 'init' sea '__init__')

    async def listFiles(self):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedSendFile}
        
        try:
            response = await self.sess.get(
                f"{self.url}/event/{self.node}/manage/attachments",
                follow_redirects=True
            )
            response.raise_for_status() # Asegurarse que la petición fue exitosa
            
            soup = BeautifulSoup(response.text, 'html.parser')
            attachments = soup.select('a.attachment[data-attachment-id]')
            listFiles = []
            
            for a in attachments:
                name = a.get_text(strip=True)
                fileID = a.get('data-attachment-id')
                if name and fileID and not name.startswith('data-'): # Filtrar nombres que quizás son metadatos
                    download_url = f"{self.url}/event/{self.node}/attachments/202/{fileID}/file"
                    listFiles.append(File(name=name, fileID=fileID, download_url=download_url))
            
            return {"listFiles": listFiles, "response": Response.SuccessSendFile}
        except httpx.HTTPStatusError as e:
            logger.error(f"Error HTTP listando archivos: {e.response.status_code} - {e}")
            return {"response": Response.FailedSendFile}
        except (httpx.RequestError, KeyError) as e:
            logger.error(f"Error listando archivos: {e}")
            return {"response": Response.FailedSendFile}
        except Exception as e:
            logger.error(f"Error inesperado listando archivos: {e}")
            return {"response": Response.FailedSendFile}

    async def deleteFile(self, identifier):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return Response.FailedDeleteFile
        
        try:
            # Extraer ID
            fileID = None
            if identifier.isdigit():
                fileID = identifier
            elif '/attachments/' in identifier:
                parts = identifier.split('/')
                for i, part in enumerate(parts):
                    if part == 'attachments' and i + 2 < len(parts):
                        fileID = parts[i + 2]
                        break
            
            if not fileID:
                logger.warning(f"No se pudo extraer el fileID del identificador: {identifier}")
                return Response.NoExistFile
            
            # Indico a veces requiere el CSRF Token para DELETE
            # Asegurarse de que el self.sess.headers se actualice con el token si es necesario
            # self.sess.headers.update({"X-Csrf-Token": self.csrfToken}) # Esto debería ser manejado globalmente por la sesión
            
            dele = await self.sess.delete(
                f"{self.url}/event/{self.node}/manage/attachments/202/{fileID}/",
                follow_redirects=True
            )
            dele.raise_for_status() # Asegurarse que la petición fue exitosa
            
            # Indico responde con JSON para DELETE
            dele_json = dele.json()
            if dele_json.get('success'):
                logger.info(f"Archivo con ID {fileID} eliminado exitosamente.")
                return Response.SuccessDeleteFile
            elif dele_json.get('error') == "No se ha podido encontrar el ítem":
                 logger.warning(f"Intento de eliminar archivo ID {fileID} fallido: no existe.")
                 return Response.NoExistFile
            else:
                logger.warning(f"Error API al eliminar archivo ID {fileID}: {dele_json}")
                return Response.FailedDeleteFile
        except httpx.HTTPStatusError as e:
            # Si el servidor responde 404, podría ser que el archivo no existe
            if e.response.status_code == 404:
                return Response.NoExistFile
            logger.error(f"Error HTTP eliminando archivo ID {fileID}: {e.response.status_code} - {e}")
            return Response.FailedDeleteFile
        except (httpx.RequestError, KeyError, ValueError) as e:
            logger.error(f"Error eliminando archivo ID {fileID}: {e}")
            return Response.FailedDeleteFile
        except Exception as e:
            logger.error(f"Error inesperado eliminando archivo ID {fileID}: {e}")
            return Response.FailedDeleteFile

    async def downloadFile(self, identifier):
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedDownloadFile}
        
        try:
            # Extraer ID
            fileID = None
            if identifier.isdigit():
                fileID = identifier
            elif '/attachments/' in identifier:
                parts = identifier.split('/')
                for i, part in enumerate(parts):
                    if part == 'attachments' and i + 2 < len(parts):
                        fileID = parts[i + 2]
                        break
            
            if not fileID:
                logger.warning(f"No se pudo extraer el fileID del identificador: {identifier}")
                return {"response": Response.FailedDownloadFile}
            
            download_url = f"{self.url}/event/{self.node}/attachments/202/{fileID}/file"
            
            async with self.sess.stream("GET", download_url, follow_redirects=True) as streamo:
                streamo.raise_for_status()
                
                disposition = streamo.headers.get("Content-Disposition", "")
                name = f"archivo_{fileID}" # Nombre por defecto
                if 'filename="' in disposition:
                    # Parsear Content-Disposition de forma más robusta
                    match = re.search(r'filename\*?=(?:UTF-8\'\'|\")?([^;\"]+)', disposition, re.IGNORECASE)
                    if match:
                        try:
                            # Decodificar URL si está codificado (RFC 5987)
                            name = urllib.parse.unquote(match.group(1))
                        except Exception:
                            name = match.group(1)
                
                local_path = os.path.join(UPLOAD_FOLDER, name)
                
                with open(local_path, "wb") as wfile:
                    async for chunk in streamo.aiter_bytes(chunk_size=8192):
                        wfile.write(chunk)
                
                logger.info(f"Archivo ID {fileID} descargado localmente como {local_path}.")
                return {"file": File(name=name, fileID=fileID, download_url=download_url), 
                        "response": Response.SuccessDownloadFile}
        except httpx.HTTPStatusError as e:
            logger.error(f"Error HTTP descargando archivo ID {fileID}: {e.response.status_code} - {e}")
            return {"response": Response.FailedDownloadFile}
        except (httpx.RequestError, FileNotFoundError, KeyError, ValueError) as e:
            logger.error(f"Error descargando archivo ID {fileID}: {e}")
            return {"response": Response.FailedDownloadFile}
        except Exception as e:
            logger.error(f"Error inesperado descargando archivo ID {fileID}: {e}")
            return {"response": Response.FailedDownloadFile}

# Descargador de enlaces externos (ya estaba en tu código original, lo mantengo aquí)
class LinkDownloader:
    @staticmethod
    def get_filename_from_url(url, content_type=None, default_name="downloaded_file"):
        """Obtener nombre de archivo desde URL"""
        parsed = urlparse(url)
        filename = os.path.basename(parsed.path)
        
        if not filename or filename == '/' or '.' not in filename: # Si no hay nombre o no hay extensión
            filename = f"{default_name}_{hash(url) % 10000}"
        
        # Asegurar extensión
        if '.' not in filename:
            if content_type:
                ext = mimetypes.guess_extension(content_type.split(';')[0]) # Split para limpiar charset
                if ext:
                    filename += ext
        
        return filename
    
    @staticmethod
    async def download_from_link(url):
        """Descargar archivo desde cualquier enlace"""
        try:
            # Se usa un nuevo AsyncClient para cada descarga para evitar problemas con la sesión del WebClient
            async with httpx.AsyncClient(timeout=30.0, follow_redirects=True) as client:
                # Hacer HEAD request primero para obtener info
                head_response = await client.head(url)
                head_response.raise_for_status()
                content_type = head_response.headers.get('Content-Type', '')
                
                filename = LinkDownloader.get_filename_from_url(url, content_type)
                local_path = os.path.join(UPLOAD_FOLDER, filename)
                
                # Descargar archivo
                async with client.stream('GET', url) as response:
                    response.raise_for_status()
                    
                    with open(local_path, 'wb') as f:
                        async for chunk in response.aiter_bytes(chunk_size=8192):
                            f.write(chunk)
                
                logger.info(f"Enlace {url} descargado localmente como {local_path}.")
                return {"file": File(name=local_path, fileID=None, download_url=url), 
                        "response": Response.SuccessDownloadLink}
                
        except httpx.HTTPStatusError as e:
            logger.error(f"Error HTTP descargando enlace {url}: {e.response.status_code} - {e}")
            return {"response": Response.FailedDownloadLink}
        except (httpx.RequestError, FileNotFoundError, KeyError, ValueError) as e:
            logger.error(f"Error descargando enlace {url}: {e}")
            return {"response": Response.FailedDownloadLink}
        except Exception as e:
            logger.error(f"Error inesperado descargando enlace {url}: {e}")
            return {"response": Response.FailedDownloadLink}

web_client = WebClient()
link_downloader = LinkDownloader()

# --- Telegram Bot Handlers ---

async def start(update: Update, context: CallbackContext):
    await update.message.reply_text(
        "🚀 BIENVENIDO AL BOT DE ARCHIVOS UCF\n\n"
        "Envía archivos grandes y los cortaré en partes de 10 MB.\n"
        "Usa /help para más información."
    )

async def help_command(update: Update, context: CallbackContext):
    help_text = """
📚 *GUÍA DE USO COMPLETA DEL BOT*

Este bot te permite gestionar archivos en la plataforma web de talleres.ucf.edu.cu.
Los archivos grandes se subirán automáticamente en partes de 10MB.

---
📤 *¿CÓMO SUBIR ARCHIVOS?*

1. *Archivos Directos:* Simplemente adjunta cualquier archivo (APK, ZIP, PDF, etc.) y envíalo al chat. El bot lo procesará y subirá a la web.
   _Archivos > 10MB se cortarán automáticamente._

2. *Enlaces Externos:* Pega un enlace de descarga directa (ej: `https://sitio.com/archivo.zip`). El bot lo descargará, lo cortará si es necesario y lo subirá a la web.

---
📋 *COMANDOS DISPONIBLES*

🔹 `/list` - Muestra una lista de todos los archivos que están actualmente en el servidor web. Incluye el Nombre y el ID necesario para otros comandos.

🔹 `/download <ID>` - Descarga un archivo del servidor web a tu bot y te lo reenvía.
    _Ejemplo: /download 4502_

🔹 `/delete <ID>` - Elimina permanentemente el archivo del servidor web. 
    *Nota:* Esta acción no se puede deshacer.
    _Ejemplo: /delete 4502_

🔹 `/start` - Reinicia el bot y muestra el mensaje de bienvenida.

---
⚠️ *NOTAS IMPORTANTES*
• Si subes un archivo cortado, necesitarás descargar TODAS las partes y unirlas en tu PC para obtener el archivo original.
• El bot se conecta con las credenciales configuradas internamente.
• El tiempo de respuesta depende del tamaño del archivo y de la velocidad de conexión del bot y la web.
    """
    await update.message.reply_text(help_text, parse_mode='Markdown')

async def list_command(update: Update, context: CallbackContext):
    await update.message.reply_text("🔍 Obteniendo lista de archivos...")
    result = await web_client.listFiles()
    
    if result["response"] == Response.FailedSendFile:
        await update.message.reply_text("❌ Error al conectar o listar archivos.")
        return
    
    files = result.get("listFiles")
    if not files:
        await update.message.reply_text("📭 No hay archivos almacenados.")
        return
    
    message = "📁 *ARCHIVOS ALMACENADOS:*\n\n"
    for f in files:
        message += f"📄 *{f.name}*\nID: `{f.id}`\n🔗 {f.download_url}\n🗑 /delete `{f.id}`\n\n"
    
    await update.message.reply_text(message, parse_mode='Markdown')

async def delete_command(update: Update, context: CallbackContext):
    if not context.args:
        await update.message.reply_text("Uso: /delete <id_o_url>\nEjemplo: /delete `12345`", parse_mode='Markdown')
        return
    
    identifier = context.args[0]
    await update.message.reply_text(f"🗑 Eliminando `{identifier}`...", parse_mode='Markdown')
    
    result = await web_client.deleteFile(identifier)
    
    if result == Response.SuccessDeleteFile:
        await update.message.reply_text("✅ Archivo eliminado.")
    elif result == Response.NoExistFile:
        await update.message.reply_text("❌ Archivo no encontrado.")
    else:
        await update.message.reply_text("❌ Error al eliminar el archivo.")

async def download_command(update: Update, context: CallbackContext):
    if not context.args:
        await update.message.reply_text("Uso: /download <id_o_url>\nEjemplo: /download `12345`", parse_mode='Markdown')
        return
    
    identifier = context.args[0]
    await update.message.reply_text(f"📥 Descargando `{identifier}` del servidor...", parse_mode='Markdown')
    
    status_msg = await update.message.reply_text("Preparando descarga...")
    await update.message.chat.send_action(ChatAction.UPLOAD_DOCUMENT)
    
    result = await web_client.downloadFile(identifier)
    
    if result["response"] == Response.SuccessDownloadFile:
        file_path_to_send = result["file"].name
        if os.path.exists(file_path_to_send):
            try:
                with open(file_path_to_send, 'rb') as file_to_send:
                    # Telegram tiene un límite de 50MB para documentos enviados por la API
                    # Si el archivo descargado es muy grande, deberías cortarlo para reenviar
                    # Para el ejercicio, lo enviamos directamente, pero considera el límite.
                    await update.message.reply_document(
                        document=file_to_send,
                        caption=f"✅ Descargado:\n📝 {result['file'].name}\n🆔 {result['file'].id}\n🔗 {result['file'].download_url}"
                    )
                await status_msg.delete() # Eliminar el mensaje de "Preparando descarga..."
                logger.info(f"Archivo {result['file'].name} enviado al usuario.")
            except Exception as e:
                await update.message.reply_text(f"❌ Error al enviar el archivo a Telegram: {e}")
                logger.error(f"Error al enviar el archivo {file_path_to_send} a Telegram: {e}")
            finally:
                os.remove(file_path_to_send) # Limpiar archivo local después de enviar
        else:
            await update.message.reply_text("❌ El archivo descargado no se encontró localmente.")
            logger.error(f"Archivo descargado '{file_path_to_send}' no encontrado para enviar.")
    else:
        await status_msg.edit_text("❌ Error al descargar el archivo del servidor web.")

async def handle_message(update: Update, context: CallbackContext):
    """Maneja cualquier mensaje: archivos o enlaces"""
    try:
        # --- Lógica de Detección de URL ---
        text = update.message.text or update.message.caption or ""
        
        # Expresiones regulares para detectar URLs
        url_patterns = [
            r'https?://[^\s]+', # http o https
            r'www\.[^\s]+\.[^\s]+', # www.dominio.com
            r'ftp://[^\s]+' # ftp
        ]
        
        is_url = False
        detected_url = None
        for pattern in url_patterns:
            match = re.search(pattern, text)
            if match:
                is_url = True
                detected_url = match.group(0) # Captura la URL encontrada
                break
        
        if is_url and detected_url:
            await update.message.reply_text(f"🔗 Detectado enlace: {detected_url[:80]}...")
            status_msg = await update.message.reply_text("Descargando desde enlace externo...")
            await update.message.chat.send_action(ChatAction.TYPING)
            
            # Descargar desde enlace
            link_result = await link_downloader.download_from_link(detected_url)
            
            if link_result["response"] == Response.SuccessDownloadLink:
                await status_msg.edit_text("✅ Descarga completada. Subiendo a la web (cortando si es necesario)...")
                await update.message.chat.send_action(ChatAction.UPLOAD_DOCUMENT)
                
                # Subir a la web, ahora sendFile manejará el corte
                web_result = await web_client.sendFile(link_result["file"].name)
                
                if web_result["response"] == Response.SuccessSendFile and web_result.get("files"):
                    uploaded_files = web_result["files"]
                    message = f"✅ ARCHIVO(S) SUBIDO(S) DESDE ENLACE\n\n"
                    if len(uploaded_files) > 1:
                        message += f"📦 Se subieron {len(uploaded_files)} partes.\n"
                    
                    for i, file_obj in enumerate(uploaded_files):
                        message += f"\n--- Parte {i+1} ---\n" if len(uploaded_files) > 1 else ""
                        message += f"📝 Nombre: {file_obj.name}\n"
                        message += f"🆔 ID: `{file_obj.id}`\n"
                        message += f"🔗 Enlace: {file_obj.download_url}\n"
                        message += f"🗑 /delete `{file_obj.id}`\n"
                        message += f"📥 /download `{file_obj.id}`\n"
                    
                    await update.message.reply_text(message, parse_mode='Markdown')
                else:
                    await update.message.reply_text("❌ Error al subir a la web.")
                
                # Limpiar archivo temporal descargado del enlace
                if os.path.exists(link_result["file"].name):
                    try:
                        os.remove(link_result["file"].name)
                    except Exception as cleanup_error:
                        logger.error(f"Error al eliminar archivo temporal descargado de enlace: {cleanup_error}")
            else:
                await update.message.reply_text("❌ Error al descargar desde el enlace.")
            
            return
        
        # --- Lógica de Archivos de Telegram ---
        file_obj = None
        file_name = None
        
        if update.message.document:
            file_obj = update.message.document
            file_name = file_obj.file_name or f"document_{file_obj.file_unique_id[:8]}"
            
        elif update.message.photo:
            file_obj = update.message.photo[-1]
            file_name = f"photo_{file_obj.file_unique_id[:8]}.jpg"
            
        elif update.message.video:
            file_obj = update.message.video
            file_name = file_obj.file_name or f"video_{file_obj.file_unique_id[:8]}.mp4"
            
        elif update.message.audio:
            file_obj = update.message.audio
            file_name = file_obj.file_name or f"audio_{file_obj.file_unique_id[:8]}.mp3"
            
        elif update.message.voice:
            file_obj = update.message.voice
            file_name = f"voice_{file_obj.file_unique_id[:8]}.ogg"
            
        elif update.message.video_note:
            file_obj = update.message.video_note
            file_name = f"video_note_{file_obj.file_unique_id[:8]}.mp4"
            
        elif update.message.sticker:
            file_obj = update.message.sticker
            file_name = f"sticker_{file_obj.file_unique_id[:8]}.webp"
        
        elif update.message.animation:  # GIFs
            file_obj = update.message.animation
            file_name = file_obj.file_name or f"animation_{file_obj.file_unique_id[:8]}.mp4"
            
        else:
            await update.message.reply_text("Envía un archivo o un enlace de descarga directa. Usa /help para ayuda.", parse_mode='Markdown')
            return
        
        # Procesar archivo de Telegram
        status_msg = await update.message.reply_text(f"📥 Descargando '{file_name}' de Telegram...")
        await update.message.chat.send_action(ChatAction.TYPING)
        
        local_path = os.path.join(UPLOAD_FOLDER, file_name)
        
        try:
            tg_file = await context.bot.get_file(file_obj.file_id)
            await tg_file.download_to_drive(local_path)
            
            await status_msg.edit_text(f"✅ Descarga de Telegram completada. Subiendo a la web (cortando si es necesario)...")
            await update.message.chat.send_action(ChatAction.UPLOAD_DOCUMENT)
            
            # Subir a la web, sendFile ahora manejará el corte
            web_result = await web_client.sendFile(local_path)
            
            if web_result["response"] == Response.SuccessSendFile and web_result.get("files"):
                uploaded_files = web_result["files"]
                message = f"✅ ARCHIVO(S) SUBIDO(S) DESDE TELEGRAM\n\n"
                if len(uploaded_files) > 1:
                    message += f"📦 Se subieron {len(uploaded_files)} partes.\n"
                
                for i, file_obj in enumerate(uploaded_files):
                    message += f"\n--- Parte {i+1} ---\n" if len(uploaded_files) > 1 else ""
                    message += f"📝 Nombre: {file_obj.name}\n"
                    message += f"🆔 ID: `{file_obj.id}`\n"
                    message += f"🔗 Enlace: {file_obj.download_url}\n"
                    message += f"🗑 /delete `{file_obj.id}`\n"
                    message += f"📥 /download `{file_obj.id}`\n"
                
                await update.message.reply_text(message, parse_mode='Markdown')
            else:
                await update.message.reply_text("❌ Error al subir a la web.")
            
        except Exception as e:
            await status_msg.edit_text(f"❌ Error durante el procesamiento: {str(e)}")
            logger.error(f"Error procesando mensaje de archivo: {e}", exc_info=True)
            
        finally:
            # Limpiar archivo temporal
            if os.path.exists(local_path):
                try:
                    os.remove(local_path)
                    logger.info(f"Archivo local {local_path} eliminado.")
                except Exception as cleanup_error:
                    logger.error(f"Error al eliminar archivo temporal: {cleanup_error}")
        
        return

def main():
    application = Application.builder().token(TELEGRAM_BOT_TOKEN).build()
    
    # Comandos
    application.add_handler(CommandHandler("start", start))
    application.add_handler(CommandHandler("help", help_command))
    application.add_handler(CommandHandler("list", list_command))
    application.add_handler(CommandHandler("delete", delete_command))
    application.add_handler(CommandHandler("download", download_command))
    
    # Manejador universal para TODO (archivos y texto con URLs)
    application.add_handler(MessageHandler(
        filters.TEXT | filters.Document.ALL | filters.PHOTO | 
        filters.VIDEO | filters.AUDIO | filters.VOICE | 
        filters.VIDEO_NOTE | filters.ANIMATION | filters.STICKER, 
        handle_message
    ))
    
    # Manejador de errores
    application.add_error_handler(lambda u, c: logger.error(f"Error: {c.error}", exc_info=True))
    
    print("="*60)
    print("🤖 BOT DE ARCHIVOS - VERSION CORTES DE 10MB")
    print("="*60)
    print("✅ Acepta TODOS los archivos y enlaces")
    print("✅ Archivos grandes se cortan en partes de 10MB")
    print("✅ Listo para recibir archivos...")
    print("="*60)
    
    application.run_polling()

if __name__ == "__main__":
    main()
