import os
import io
import math
import json
import logging
import mimetypes
import httpx
import asyncio

from bs4 import BeautifulSoup
from telegram import Update
from telegram.constants import ChatAction
from telegram.ext import Application, CommandHandler, MessageHandler, filters, ContextTypes

# Configuración
logging.basicConfig(format='%(asctime)s - %(name)s - %(levelname)s - %(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)

# Usa variables de entorno en lugar de hardcodear tokens/credenciales
TELEGRAM_BOT_TOKEN = os.environ.get("TELEGRAM_BOT_TOKEN", "7971082679:AAFNvmUMFjQ2utlessC2-b-yjiFBN-c16bA")
UPLOAD_FOLDER = "downloads"
os.makedirs(UPLOAD_FOLDER, exist_ok=True)

CHUNK_SIZE = 10 * 1024 * 1024  # 10 MB

class Response:
    SuccessLogin = "SuccessLogin"
    FailedLogin = "FailedLogin"
    SuccessSendFile = "SuccessSendFile"
    FailedSendFile = "FailedSendFile"
    SuccessDeleteFile = "SuccessDeleteFile"
    FailedDeleteFile = "FailedDeleteFile"
    NoExistFile = "NoExistFile"
    SuccessDownloadFile = "SuccessDownloadFile"
    FailedDownloadFile = "FailedDownloadFile"
    SuccessDownloadLink = "SuccessDownloadLink"

class WebClient:
    def __init__(self, base_url, node, username, password):
        self.url = base_url.rstrip("/")
        self.node = node
        self.username = username
        self.password = password
        self.connected = False
        self.sess = httpx.AsyncClient(timeout=None)
        self.csrfToken = None

    async def connect(self):
        try:
            form = await self.sess.get(f"{self.url}/login/indico/form")
            soup = BeautifulSoup(form.json()["html"], "html.parser")
            token_elem = soup.find('input', {'name': 'csrf_token'})
            if token_elem is None:
                logger.error("No se encontró csrf_token en el formulario")
                return Response.FailedLogin
            self.csrfToken = token_elem['value']

            login = await self.sess.post(
                f"{self.url}/login",
                data={
                    "csrf_token": self.csrfToken,
                    "_provider": "indico",
                    "identifier": self.username,
                    "password": self.password
                },
                follow_redirects=True
            )

            if "Cerrar sesión" in login.text:
                self.connected = True
                return Response.SuccessLogin
            logger.error("Inicio de sesión fallido, respuesta HTML no contiene 'Cerrar sesión'")
            return Response.FailedLogin
        except Exception as e:
            logger.exception(f"Error en conexión: {e}")
            return Response.FailedLogin

    async def sendFile(self, path):
        """
        Divide el archivo en partes de CHUNK_SIZE y sube cada parte con el campo "files".
        Al final sube un manifest JSON con la lista de partes subidas.
        """
        if not self.connected:
            response = await self.connect()
            if response == Response.FailedLogin:
                return {"response": Response.FailedSendFile, "detail": "login_failed"}

        try:
            total_size = os.path.getsize(path)
            parts = max(1, math.ceil(total_size / CHUNK_SIZE))
            basename = os.path.basename(path)
            mime_type = mimetypes.guess_type(path)[0] or "application/octet-stream"
            uploadedparts = []

            with open(path, "rb") as f:
                for i in range(parts):
                    chunkbytes = f.read(CHUNKSIZE)
                    if not chunkbytes:
                        break
                    partindex = i + 1
                    partname = f"{basename}.part{str(partindex).zfill(3)}"
                    fileobj = io.BytesIO(chunkbytes)
                    # Reiniciar puntero al inicio del BytesIO
                    fileobj.seek(0)

                    # Construir data y files para la petición (igual que antes)
                    data = {
                        "csrftoken": self.csrfToken,
                        "filechangetrigger": "added-file",
                        "folder": "None",
                        "acl": []
                    }

                    files = {"files": (partname, fileobj, mimetype)}
                    up = await self.sess.post(
                        f"{self.url}/event/{self.node}/manage/attachments/add/files",
                        data=data,
                        files=files,
                        followredirects=True
                    )

                    fileobj.close()

                    # Comprobar respuesta
                    if up.statuscode != 200:
                        logger.error(f"Error subida parte {partname}: status={up.statuscode}")
                        return {"response": Response.FailedSendFile, "detail": f"status{up.statuscode}", "part": partname}

                    try:
                        j = up.json()
                    except Exception:
                        logger.exception("Respuesta no JSON al subir parte")
                        return {"response": Response.FailedSendFile, "detail": "invalidjsonresponse", "part": partname}

                    if not j.get("success"):
                        logger.error(f"Upload part failed: {j}")
                        return {"response": Response.FailedSendFile, "detail": j, "part": partname}

                    # Parse attachmentlist para obtener id si es necesario
                    soup = BeautifulSoup(j.get("attachmentlist", ""), "html.parser")
                    attachments = soup.select('a.attachmentdata-attachment-id')
                    # Buscar la parte recién subida por nombre
                    foundid = None
                    for a in attachments:
                        name = a.gettext(strip=True)
                        fileID = a.get('data-attachment-id')
                        if name == partname and fileID:
                            foundid = fileID
                            break

                    uploadedparts.append({"partname": partname, "attachmentid": foundid})

            # Subir manifest JSON con la lista de partes (opcional)
            manifest = {
                "originalname": basename,
                "size": totalsize,
                "partscount": len(uploadedparts),
                "parts": uploadedparts
            }
            manifestbytes = json.dumps(manifest, ensureascii=False).encode("utf-8")
            manifestname = f"{basename}.manifest.json"
            manifestfileobj = io.BytesIO(manifestbytes)
            manifestfileobj.seek(0)

            data = {
                "csrftoken": self.csrfToken,
                "filechangetrigger": "added-file",
                "folder": "None",
                "acl": []
            }
            files = {"files": (manifestname, manifestfileobj, "application/json")}
            upm = await self.sess.post(
                f"{self.url}/event/{self.node}/manage/attachments/add/files",
                data=data,
                files=files,
                followredirects=True
            )
            manifestfileobj.close()

            if upm.statuscode != 200:
                logger.warning("No se pudo subir el manifest, pero las partes pueden haberse subido correctamente.")
                return {"response": Response.SuccessSendFile, "detail": "partsuploadedbutmanifestfailed", "parts":uploaded_parts}

            return {"response": Response.SuccessSendFile, "parts": uploaded_parts}

        except Exception as e:
            logger.exception(f"Error al enviar archivo: {e}")
            return {"response": Response.FailedSendFile, "detail": str(e)}

# Instancia del cliente web (rellena credenciales vía variables de entorno)
WEB_BASE_URL = os.environ.get("WEB_BASE_URL", "https://eventos.uh.cu/")
WEB_NODE = int(os.environ.get("WEB_NODE", "150"))
WEB_USERNAME = os.environ.get("WEB_USERNAME", "luisernestorb95")
WEB_PASSWORD = os.environ.get("WEB_PASSWORD", "Luisito1995*")

webclient = WebClient(WEB_BASE_URL, WEB_NODE, WEB_USERNAME, WEB_PASSWORD)

# ---------- Handlers del bot ----------
async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text("Hola. Envíame un archivo y lo subiré (se dividirá en partes de 10 MB si es necesario).")

async def handle_any_file(update: Update, context: ContextTypes.DEFAULT_TYPE):
    msg = update.message
    if msg is None:
        return

    # Determinar file_id y nombre
    file_id = None
    filename = None

    if msg.document:
        file_id = msg.document.file_id
        filename = msg.document.file_name or f"document_{file_id}"
    elif msg.audio:
        file_id = msg.audio.file_id
        filename = msg.audio.file_name or f"audio_{file_id}"
    elif msg.video:
        file_id = msg.video.file_id
        filename = msg.video.file_name or f"video_{file_id}"
    elif msg.voice:
        file_id = msg.voice.file_id
        filename = f"voice_{file_id}.ogg"
    elif msg.animation:
        file_id = msg.animation.file_id
        filename = msg.animation.file_name or f"animation_{file_id}"
    elif msg.photo:
        # photo es una lista por resoluciones; coger la mayor
        photo = msg.photo[-1]
        file_id = photo.file_id
        filename = f"photo_{file_id}.jpg"
    elif msg.sticker:
        file_id = msg.sticker.file_id
        filename = f"sticker_{file_id}.webp"
    else:
        await msg.reply_text("No se ha detectado un archivo compatible.")
        return

    # Descargar el archivo al disco
    try:
        await context.bot.send_chat_action(chat_id=msg.chat_id, action=ChatAction.UPLOAD_DOCUMENT)
        telegram_file = await context.bot.get_file(file_id)
        safe_name = filename.replace("/", "_")
        path = os.path.join(UPLOAD_FOLDER, safe_name)
        await telegram_file.download_to_drive(custom_path=path)
        await msg.reply_text(f"Archivo recibido: {safe_name}. Subiendo al servidor (puede tardar).")
    except Exception as e:
        logger.exception("Error descargando archivo de Telegram")
        await msg.reply_text(f"Error al descargar el archivo: {e}")
        return

    # Enviar al servidor (con división en partes)
    try:
        result = await webclient.sendFile(path)
        if result.get("response") == Response.SuccessSendFile:
            parts = result.get("parts", [])
            await msg.reply_text(f"Subida completada. Partes: {len(parts)}")
        else:
            await msg.reply_text(f"Error subiendo archivo: {result}")
    except Exception as e:
        logger.exception("Error subiendo archivo al servidor")
        await msg.reply_text(f"Error al subir archivo al servidor: {e}")

async def main():
    app = Application.builder().token(TELEGRAM_BOT_TOKEN).build()

    app.add_handler(CommandHandler("start", start))
    # Filter para casi todo lo que es archivo/media
    filter_files = (
        filters.Document.ALL
        | filters.AUDIO
        | filters.VIDEO
        | filters.VOICE
        | filters.PHOTO
        | filters.ANIMATION
        | filters.STICKER
    )
    app.add_handler(MessageHandler(filter_files, handle_any_file))

    logger.info("Bot iniciado")
    await app.run_polling()

if __name__ == "__main__":
    asyncio.run(main())